package jp.co.ois.hoge.springsecurity.config;

import java.io.IOException;
import java.security.SecureRandom;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.crypto.factory.PasswordEncoderFactories;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.header.HeaderWriterFilter;
import org.springframework.ui.Model;
import org.springframework.util.Base64Utils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ModelAttribute;

@Configuration
@EnableWebSecurity
public class WebSecurityConfig {

    @Bean
    public SecurityFilterChain securityFilterChain(HttpSecurity http) throws Exception {
        http
                .authorizeHttpRequests(requests -> requests
                        .antMatchers("/static/**", "/login*", "/logout*", "/hoge").permitAll()
                        .anyRequest().authenticated())
                .formLogin(formLogin -> formLogin
                        .usernameParameter("username")
                        .passwordParameter("password")
                        .loginPage("/login")
                        .successForwardUrl("/login-success")
                        .failureForwardUrl("/login-failure")
                        .permitAll())
                .logout(logout -> logout
                        .deleteCookies()
                        .invalidateHttpSession(true)
                        .logoutUrl("/logout")
                        .logoutSuccessUrl("/login")
                        .permitAll())
                .headers(headers -> headers
                        .contentSecurityPolicy(csp -> csp
                                .policyDirectives("default-src 'self' 'nonce-{nonce}';"
                                        + " frame-ancestors 'none'; form-action 'self';")
                                ))
                .addFilterBefore(new CspNonceFilter(), HeaderWriterFilter.class)
                .exceptionHandling(handling -> handling
                        .accessDeniedPage("/error/403"));
        return http.build();
    }

    @Bean
    PasswordEncoder passwordEncoder() {
        return PasswordEncoderFactories.createDelegatingPasswordEncoder();
    }

    public static class CspNonceFilter implements Filter {
        private static final SecureRandom secureRandom = new SecureRandom();

        @Override
        public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
                throws IOException, ServletException {
            byte[] nonceBytes = new byte[16];
            secureRandom.nextBytes(nonceBytes);
            String nonce = Base64Utils.encodeToString(nonceBytes);
            request.setAttribute("cspNonce", nonce);
            chain.doFilter(request, new CspNonceResponseWrapper((HttpServletResponse) response, nonce));
        }
    }
    
    public static class CspNonceResponseWrapper extends HttpServletResponseWrapper {
        private final String nonce;

        public CspNonceResponseWrapper(HttpServletResponse response, String nonce) {
            super(response);
            this.nonce = nonce;
        }

        private String getHeaderValue(String name, String value) {
            if (name.equals("Content-Security-Policy") && StringUtils.hasText(value)) {
                return value.replace("{nonce}", nonce);
            } else {
                return value;
            }
        }

        @Override
        public void setHeader(String name, String value) {
            super.setHeader(name, getHeaderValue(name, value));
        }

        @Override
        public void addHeader(String name, String value) {
            super.addHeader(name, getHeaderValue(name, value));
        }
    }
    
    @ControllerAdvice
    public static class CspNonceControllerAdvice {
        @ModelAttribute
        public void addAttributes(Model model, HttpServletRequest request) {
            model.addAttribute("nonce", request.getAttribute("cspNonce"));
        }
    }
}
